<?php

namespace App\Lib;

use App\Constants\Status;
use App\Models\Platform;
use App\Models\SocialAccount;
use Illuminate\Support\Facades\Config;
use Laravel\Socialite\Facades\Socialite;

class SocialConnect {
    private $provider;

    public function __construct($provider) {
        $this->provider = $provider;
        $this->configuration();
    }

    public function redirectDriver() {
        if ($this->provider == 'facebook') {
            $provider    = 'facebook';
            $permissions = [
                'pages_manage_posts',
                'pages_read_engagement',
                'pages_show_list',
                'pages_manage_metadata',
                'business_management',
                'pages_manage_engagement',
                'pages_read_user_content',
                'public_profile',
            ];
        } else if ($this->provider == 'instagram') {
            $provider    = 'facebook';
            $permissions = [
                'instagram_basic',
                'pages_show_list',
                'business_management',
                'instagram_manage_messages',
                'pages_read_engagement',
            ];
        } else if ($this->provider == 'linkedin-openid') {
            $provider    = 'linkedin-openid';
            $permissions = [
                'openid',
                'profile',
                'email',
                'w_member_social',
                'r_events',
                'rw_events',
            ];
        } else if ($this->provider == 'twitter') {
            $provider    = 'twitter';
            $permissions = [
                'tweet.read',
                'tweet.write',
                'users.read',
                'offline.access',
                '+media.write',
            ];
        } else if ($this->provider == 'tiktok') {
            $provider    = 'tiktok';
            $permissions = [
                'user.info.basic',
                'user.info.profile',
                'user.info.stats',
                'video.list',
                'video.upload',       
                'video.publish',  
            ];
        } else {
            $provider    = 'google';
            $permissions = [
                'https://www.googleapis.com/auth/youtube',
                'https://www.googleapis.com/auth/youtube.readonly',
                'https://www.googleapis.com/auth/youtube.force-ssl',
            ];
        }

        if ($provider === 'google') {
            $driver = Socialite::driver($provider)->scopes($permissions);
            $driver->with([
                'access_type' => 'offline',
                'prompt'      => 'consent',
            ]);
            return $driver->redirect();
        } else if ($provider === 'twitter') {
            return Socialite::driver('twitter-oauth-2')->scopes($permissions)->redirect();
        } else {
            return Socialite::driver($provider)->scopes($permissions)->redirect();
        }
    }

    private function configuration() {
        if ($this->provider == 'youtube') {
            $provider      = 'google';
            $configuration = gs()->social_connect_credential->youtube;
        } else if ($this->provider == 'linkedin-openid') {
            $provider      = 'linkedin-openid';
            $configuration = gs()->social_connect_credential->linkedin;
        } else if ($this->provider == 'twitter') {
            $provider      = 'twitter';
            $configuration = gs()->social_connect_credential->twitter;
        } else if ($this->provider == 'tiktok') {
            $provider      = 'tiktok';
            $configuration = gs()->social_connect_credential->tiktok;
        } else {
            $provider      = 'facebook';
            $configuration = gs()->social_connect_credential->$provider;
        }
        Config::set('services.' . $provider, [
            'client_id'     => $configuration->client_id,
            'client_secret' => $configuration->client_secret,
            'redirect'      => route("user.social.connect.callback", $this->provider),
        ]);
    }

    public function login() {

        if ($this->provider == 'linkedin-openid') {
            $provider = 'linkedin-openid';
        } else if ($this->provider == 'youtube') {
            $provider = 'google';
        } else if ($this->provider == 'twitter') {
            $provider = 'twitter';
        } else if ($this->provider == 'tiktok') {
            $provider = 'tiktok';
        } else {
            $provider = 'facebook';
        }

        if ($provider === 'twitter') {
            $user = Socialite::driver('twitter-oauth-2')->user();
        } else {
            $user = Socialite::driver($provider)->user();
        }

        if ($this->provider == 'facebook') {
            $connect = $this->connectFacebook($user);
            if (isset($connect['error']) && $connect['error']) {
                $notify[] = ['error', $connect['message']];
                return to_route('user.social.index')->withNotify($notify);
            }
            $notification = 'Facebook page connected successfully';

        } else if ($this->provider == 'instagram') {
            $connect = $this->connectInstagram($user);
            if (isset($connect['error']) && $connect['error']) {
                $notify[] = ['error', $connect['message']];
                return to_route('user.social.index')->withNotify($notify);
            }
            $notification = 'Instagram account connected successfully';

        } else if ($this->provider == 'linkedin-openid') {
            $connect = $this->connectLinkedin($user);
            if (isset($connect['error']) && $connect['error']) {
                $notify[] = ['error', $connect['message']];
                return to_route('user.social.index')->withNotify($notify);
            }
            $notification = 'Linkedin account connected successfully';

        } else if ($this->provider == 'twitter') {
            $connect = $this->connectTwitter($user);
            if (isset($connect['error']) && $connect['error']) {
                $notify[] = ['error', $connect['message']];
                return to_route('user.social.index')->withNotify($notify);
            }
            $notification = 'Twitter account connected successfully';
        } else if ($this->provider == 'tiktok') {
            $connect = $this->connectTiktok($user);
            if (isset($connect['error']) && $connect['error']) {
                $notify[] = ['error', $connect['message']];
                return to_route('user.social.index')->withNotify($notify);
            }
            $notification = 'Twitter account connected successfully';
        } else {
            $connect = $this->connectYoutube($user);
            if (isset($connect['error']) && $connect['error']) {
                $notify[] = ['error', $connect['message']];
                return to_route('user.social.index')->withNotify($notify);
            }
            $notification = 'Youtube channel connected successfully';

        }

        $notify[] = ['success', $notification];
        return to_route('user.social.index')->withNotify($notify);
    }

    private function connectFacebook($user) {
        try {
            $data     = CurlRequest::curlContent('https://graph.facebook.com/v23.0/me/accounts?access_token=' . $user->token);
            $response = json_decode($data);
            if (!$response || !isset($response->data)) {
                return ['error' => true, 'message' => 'Failed to fetch data from Facebook'];
            }

            $platform = Platform::where('name', 'facebook')->first();
            $authUser = auth()->user();
            foreach ($response->data as $page) {
                if (SocialAccount::where('user_id', auth()->id())->where('platform_id', $platform->id)->where('profile_id', $page->id)->exists()) {
                    continue;
                }

                $authUser->connected_channel += 1;
                $authUser->save();

                $socialAccount                = new SocialAccount();
                $socialAccount->user_id       = $authUser->id;
                $socialAccount->platform_id   = $platform->id;
                $socialAccount->profile_id    = $page->id;
                $socialAccount->profile_name  = $page->name;
                $socialAccount->profile_image = 'https://graph.facebook.com/' . $page->id . '/picture?width=200&height=200';
                $socialAccount->access_token  = $page->access_token;
                $socialAccount->expires_at    = now()->addDays(30);
                $socialAccount->extra_data    = $page->name;
                $socialAccount->status        = Status::ENABLE;
                $socialAccount->save();
            }

            return ['error' => false];
        } catch (\Throwable $th) {
            return ['error' => true, 'message' => 'An error occurred while connecting Facebook pages'];
        }
    }

    private function connectInstagram($user) {
        try {
            $data     = CurlRequest::curlContent("https://graph.facebook.com/v23.0/me/accounts?fields=id,name,username,access_token,instagram_business_account&access_token=$user->token&fields=instagram_business_account,access_token");
            $response = json_decode($data);
            if (!$response || !isset($response->data)) {
                return ['error' => true, 'message' => 'Failed to fetch data from Facebook'];
            }
            $platform = Platform::where('name', 'instagram')->first();
            $authUser = auth()->user();

            if(!isset($response->data[0]->instagram_business_account)) {
                return ['error' => true, 'message' => 'No Instagram account is currently connected to the Facebook account'];
            }

            foreach ($response->data as $page) {
                if (!isset($page->instagram_business_account)) {
                    continue;
                }

                if (SocialAccount::where('user_id', auth()->id())->where('platform_id', $platform->id)->where('profile_id', $page->id)->exists()) {
                    continue;
                }

                $authUser->connected_channel += 1;
                $authUser->save();

                $businessAccountId = $response->data[0]->instagram_business_account->id;
                $instagramData     = CurlRequest::curlContent("https://graph.facebook.com/v23.0/{$businessAccountId}?fields=username,name&access_token={$user->token}");
                $instagramResponse = json_decode($instagramData);

                $socialAccount                = new SocialAccount();
                $socialAccount->user_id       = $authUser->id;
                $socialAccount->platform_id   = $platform->id;
                $socialAccount->profile_id    = $page->id;
                $socialAccount->profile_name  = "@" . $instagramResponse->username;
                $socialAccount->profile_image = 'https://graph.facebook.com/' . $page->id . '/picture?width=200&height=200';
                $socialAccount->access_token  = $page->access_token;
                $socialAccount->expires_at    = now()->addDays(30);
                $socialAccount->extra_data    = $instagramResponse;
                $socialAccount->status        = Status::ENABLE;
                $socialAccount->save();
            }
            return ['error' => false];
        } catch (\Throwable $th) {
            return ['error' => true, 'message' => 'An error occurred while connecting Instagram account'];
        }
    }

    private function connectYoutube($user) {
        try {
            $url  = 'https://youtube.googleapis.com/youtube/v3/channels?part=snippet%2CcontentDetails%2Cstatistics&mine=true&key=' . gs()->social_connect_credential->youtube->api_key;
            $data = CurlRequest::curlContent($url, [
                "Authorization: Bearer $user->token",
                'Accept: application/json',
            ]);
            $response = json_decode($data);
            if (!$response) {
                return ['error' => true, 'message' => 'Invalid API request'];
            }

            $platformId = Platform::where('name', 'youtube')->first();
            $authUser   = auth()->user();
            foreach ($response->items as $page) {
                if (SocialAccount::where('user_id', auth()->id())->where('platform_id', $platformId->id)->where('profile_id', $page->id)->exists()) {
                    continue;
                }

                $authUser->connected_channel += 1;
                $authUser->save();

                $socialAccount                = new SocialAccount();
                $socialAccount->user_id       = $authUser->id;
                $socialAccount->platform_id   = $platformId->id;
                $socialAccount->profile_id    = $page->id;
                $socialAccount->profile_name  = $page->snippet->title;
                $socialAccount->profile_image = $page->snippet->thumbnails->medium->url ?? null;
                $socialAccount->access_token  = $user->token;
                $socialAccount->refresh_token = $user->refreshToken;
                $socialAccount->expires_at    = now()->addDays(30);
                $socialAccount->extra_data    = $page->snippet;
                $socialAccount->status        = Status::ENABLE;
                $socialAccount->save();
            }

            return ['error' => false];
        } catch (\Throwable $th) {
            return ['error' => true, 'message' => 'An error occurred while connecting Youtube channel'];
        }
    }

    private function connectLinkedin($user) {
        try {
            $platform = Platform::where('name', 'linkedin-openid')->first();
            $authUser = auth()->user();

            if (SocialAccount::where('user_id', auth()->id())->where('platform_id', $platform->id)->where('profile_id', $user->id)->exists()) {
                return ['error' => true, 'message' => 'Social account already connected'];
            }

            $authUser->connected_channel += 1;
            $authUser->save();

            $socialAccount                = new SocialAccount();
            $socialAccount->user_id       = $authUser->id;
            $socialAccount->platform_id   = $platform->id;
            $socialAccount->profile_id    = $user->id;
            $socialAccount->profile_name  = $user->name;
            $socialAccount->profile_image = $user->avatar;
            $socialAccount->access_token  = $user->token;
            $socialAccount->refresh_token = $user->refreshToken;
            $socialAccount->expires_at    = now()->second($user->expiresIn);
            $socialAccount->extra_data    = $user->name;
            $socialAccount->status        = Status::ENABLE;
            $socialAccount->save();

            return ['error' => false];
        } catch (\Throwable $th) {
            return ['error' => true, 'message' => 'An error occurred while connecting Linkedin account'];
        }
    }

    private function connectTwitter($user) {
        try {
            $platform = Platform::where('name', 'twitter')->first();
            $authUser = auth()->user();

            if (SocialAccount::where('user_id', auth()->id())->where('platform_id', $platform->id)->where('profile_id', $user->id)->exists()) {
                return ['error' => true, 'message' => 'Social account already connected'];
            }

            $authUser->connected_channel += 1;
            $authUser->save();

            $socialAccount                = new SocialAccount();
            $socialAccount->user_id       = $authUser->id;
            $socialAccount->platform_id   = $platform->id;
            $socialAccount->profile_id    = $user->id;
            $socialAccount->profile_name  = $user->name;
            $socialAccount->profile_image = $user->avatar;
            $socialAccount->access_token  = $user->token;
            $socialAccount->refresh_token = $user->refreshToken;
            $socialAccount->expires_at    = now()->second($user->expiresIn);
            $socialAccount->extra_data    = $user->name;
            $socialAccount->status        = Status::ENABLE;
            $socialAccount->save();

            return ['error' => false];
        } catch (\Throwable $th) {
            return ['error' => true, 'message' => 'An error occurred while connecting Twitter account'];
        }
    }
    
    private function connectTiktok($user) {
        try {
            $platform = Platform::where('name', 'tiktok')->first();
            $authUser = auth()->user();

            if (SocialAccount::where('user_id', auth()->id())->where('platform_id', $platform->id)->where('profile_id', $user->id)->exists()) {
                return ['error' => true, 'message' => 'Social account already connected'];
            }

            $authUser->connected_channel += 1;
            $authUser->save();
            $socialAccount                = new SocialAccount();
            $socialAccount->user_id       = $authUser->id;
            $socialAccount->platform_id   = $platform->id;
            $socialAccount->profile_id    = $user->id;
            $socialAccount->profile_name  = $user->name;
            $socialAccount->profile_image = $user->avatar;
            $socialAccount->access_token  = $user->token;
            $socialAccount->refresh_token = $user->refreshToken;
            $socialAccount->expires_at    = now()->second($user->expiresIn);
            $socialAccount->extra_data    = $user->name;
            $socialAccount->status        = Status::ENABLE;
            $socialAccount->save();

            return ['error' => false];
        } catch (\Throwable $th) {
            return ['error' => true, 'message' => 'An error occurred while connecting Twitter account'];
        }
    }
}
