<?php

namespace App\Lib;

use App\Constants\Status;
use App\Models\SocialAccount;

class MetaPublisher {
    protected $post;
    public function __construct($post = null) {
        $this->post = $post;
    }

    public function publish() {
        try {
            $post       = $this->post;
            $accountIds = $post->account_ids;

            foreach ($accountIds as $accountId) {
                $socialAccount = SocialAccount::active()->where('id', $accountId)
                    ->where('user_id', $post->user_id)
                    ->first();
                if (!$socialAccount) {
                    $notify = 'Invalid Social Account';
                    userNotification($post->user_id, $notify, route('user.post.index'));
                    continue;
                }

                try {
                    switch ($socialAccount->platform_id) {
                    case Status::FACEBOOK:
                        $this->postToFacebook($post, $socialAccount);
                        break;

                    case Status::INSTAGRAM:
                        $this->postToInstagram($post, $socialAccount);
                        break;

                    case Status::YOUTUBE:
                        $this->postToYouTube($post, $socialAccount);
                        break;

                    case Status::TWITTER:
                        $this->postToTwitter($post, $socialAccount);
                        break;

                    case Status::LINKEDIN:
                        $this->postToLinkedin($post, $socialAccount);
                        break;

                    case Status::TELEGRAM:
                        $this->postToTelegram($post, $socialAccount);
                        break;
                    case Status::TIKTOK:
                        $this->postToTiktok($post, $socialAccount);
                        break;

                    default:
                        continue 2;
                    }
                } catch (\Throwable $th) {
                    $notify = "Something went wrong while posting to {$socialAccount->platform->name}";
                    userNotification($post->user_id, $notify, route('user.post.index'));
                    continue;
                }
            }
            return [
                'status'  => 'success',
                'message' => 'Post published successfully',
            ];
        } catch (\Throwable $th) {
            return [
                'status'  => 'error',
                'message' => $th->getMessage(),
            ];
        }
    }

    public function postToFacebook($post, $socialAccount) {
        $accessToken = $socialAccount->access_token;
        $pageId      = $socialAccount->profile_id;
        $content     = $post->content . ($post->tags ? "\n\n{$post->tags}" : '');
        $headers     = ['Content-Type: application/json'];
        $mediaIds    = [];

        foreach ($post->medias as $media) {
            try {
                if ($media->media_type == 'image') {
                    $imageUrl   = getImage(getFilePath('postMedia') . '/' . $media->media_path);
                    $mediaIds[] = $this->uploadImageOnFacebook($imageUrl, $pageId, $accessToken);
                } else {
                    $response = $this->uploadVideoOnFacebook($media->media_path, $pageId, $accessToken, $content);

                    if ($post->first_comment != null) {
                        $this->postComment($socialAccount, $response, $post->first_comment);
                    }
                    return true;
                }
            } catch (\Exception $e) {
                continue;
            }
        }

        $endpoint = "https://graph.facebook.com/v23.0/{$pageId}/feed";
        $body     = [
            'message'      => $content,
            'access_token' => $accessToken,
        ];

        if (!empty($mediaIds)) {
            $body['attached_media'] = array_map(fn($id) => ['media_fbid' => $id], $mediaIds);
        }

        try {
            $response = CurlRequest::curlPostContent($endpoint, $body, $headers);
            $response = json_decode($response, true);
            if ($post->first_comment != null) {
                $this->postComment($socialAccount, $response['id'], $post->first_comment);
            }
        } catch (\Exception $e) {
            info("Facebook post error: " . $e->getMessage());
        }
    }

    public function uploadImageOnFacebook($imagePath, $pageId, $accessToken) {
        try {
            $endpoint = "https://graph.facebook.com/v23.0/{$pageId}/photos";
            $headers  = ['Content-Type' => 'multipart/form-data'];
            $body     = [
                'url'          => $imagePath,
                'published'    => false,
                'access_token' => $accessToken,
            ];

            $response = CurlRequest::curlPostContent($endpoint, $body, $headers);
            $response = json_decode($response);

            return $response->id ?? null;
        } catch (\Exception $e) {
            info("Facebook media upload error: " . $e->getMessage());
        }
    }

    public function uploadVideoOnFacebook($videoPath, $pageId, $accessToken, $content) {
        try {
            $videoPath = getImage(getFilePath('postMedia') . '/' . $videoPath);
            $endpoint  = "https://graph.facebook.com/v23.0/{$pageId}/videos";

            $body = [
                'file_url'     => $videoPath,
                'published'    => true,
                'description'  => $content ?? '',
                'access_token' => $accessToken,
            ];

            $response = CurlRequest::curlPostContent($endpoint, $body);
            $response = json_decode($response, true);

            if (isset($response['id'])) {
                return $response['id'];
            }

            return false;
        } catch (\Exception $e) {
            info("Facebook media upload error: " . $e->getMessage());
            return false;
        }
    }

    public function postToInstagram($post, $socialAccount) {
        $accessToken = $socialAccount->access_token;
        $pageId      = $socialAccount->profile_id;
        $content     = $post->content . ($post->tags ? "\n\n{$post->tags}" : '');

        $businessAccountJson = CurlRequest::curlContent(
            "https://graph.facebook.com/v20.0/{$pageId}?fields=instagram_business_account&access_token={$accessToken}"
        );
        $businessAccount = json_decode($businessAccountJson);
        $businessId      = $businessAccount->instagram_business_account->id;

        if (count($post->medias) === 1) {
            $media = $post->medias[0];

            if ($media->media_type == 'video') {
                $postId = $this->uploadVideoOnInstagram($media->media_url, $content, $businessId, $accessToken);
            } else {
                $imageUrl = getImage(getFilePath('postMedia') . '/' . $media->media_path);
                $response = CurlRequest::curlPostContent(
                    "https://graph.facebook.com/v20.0/{$businessId}/media",
                    [
                        'image_url'    => $imageUrl,
                        'caption'      => $content,
                        'access_token' => $accessToken,
                    ]
                );
                $response   = json_decode($response);
                $creationId = $response->id;

                $publishResponse = CurlRequest::curlPostContent(
                    "https://graph.facebook.com/v20.0/{$businessId}/media_publish",
                    [
                        'creation_id'  => $creationId,
                        'access_token' => $accessToken,
                    ]
                );
                $publishResponse = json_decode($publishResponse);
                $postId          = $publishResponse->id;
            }

            if ($post->first_comment) {
                $this->postComment($socialAccount, $postId, $post->first_comment);
            }
            return $publishResponse;
        } else {
            $mediaIds = [];

            foreach ($post->medias as $media) {
                if ($media->media_type == 'video') {
                    continue;
                } else {
                    $imageUrl = getImage(getFilePath('postMedia') . '/' . $media->media_path);
                    $response = CurlRequest::curlPostContent(
                        "https://graph.facebook.com/v20.0/{$businessId}/media",
                        [
                            'image_url'        => $imageUrl,
                            'is_carousel_item' => true,
                            'access_token'     => $accessToken,
                        ]
                    );
                    $response   = json_decode($response);
                    $mediaIds[] = $response->id;
                }
            }

            $carouselResponse = CurlRequest::curlPostContent(
                "https://graph.facebook.com/v20.0/{$businessId}/media",
                [
                    'caption'      => $content,
                    'media_type'   => 'CAROUSEL',
                    'children'     => implode(',', $mediaIds),
                    'access_token' => $accessToken,
                ]
            );

            $carouselResponse = json_decode($carouselResponse);
            $creationId       = $carouselResponse->id;

            $publishResponse = CurlRequest::curlPostContent(
                "https://graph.facebook.com/v20.0/{$businessId}/media_publish",
                [
                    'creation_id'  => $creationId,
                    'access_token' => $accessToken,
                ]
            );

            $publishResponse = json_decode($publishResponse);
            $postId          = $publishResponse->id;

            if ($post->first_comment) {
                $this->postComment($socialAccount, $postId, $post->first_comment);
            }
            return $publishResponse;
        }
    }

    public function uploadVideoOnInstagram($videoPath, $content, $businessId, $accessToken) {
        $videoPath = getImage(getFilePath('postMedia') . '/' . $videoPath);

        try {
            $endPoint = "https://graph.facebook.com/v20.0/{$businessId}/media";
            $headers  = [
                "Content-Type" => 'application/x-www-form-urlencoded',
            ];

            $body = [
                'media_type'   => 'REELS',
                'video_url'    => $videoPath,
                'caption'      => $content,
                'access_token' => $accessToken,
            ];

            $response = CurlRequest::curlPostContent($endPoint, $body, $headers);
            $result   = json_decode($response);
            $videoId  = $result->id;

            $maxAttempts = 20;
            $attempt     = 0;
            $isReady     = false;

            while ($attempt < $maxAttempts && !$isReady) {
                sleep(5);

                $statusUrl    = "https://graph.facebook.com/v20.0/{$videoId}";
                $statusParams = [
                    'fields'       => 'status_code',
                    'access_token' => $accessToken,
                ];

                $statusResponse = CurlRequest::curlContent($statusUrl . '?' . http_build_query($statusParams));
                $statusResult   = json_decode($statusResponse);

                if (isset($statusResult->status_code)) {
                    if ($statusResult->status_code === 'FINISHED') {
                        $isReady = true;
                    } else if ($statusResult->status_code === 'ERROR') {
                        break;
                    }
                }
                $attempt++;
            }

            $publishUrl  = "https://graph.facebook.com/v20.0/{$businessId}/media_publish";
            $publishBody = [
                'creation_id'  => $videoId,
                'access_token' => $accessToken,
            ];

            $publishResponse = CurlRequest::curlPostContent($publishUrl, $publishBody, $headers);
            $publishResult   = json_decode($publishResponse);

            return $publishResult->id ?? null;
        } catch (\Throwable $th) {
        }
    }

    public function postToYouTube($post, $socialAccount) {
        $accessToken = $this->refreshYoutubeAccessToken($socialAccount);
        $description = $post->content;
        $title       = $post->title;
        $videoPath   = getImage(getFilePath('postMedia') . '/' . $post->medias->first()->media_path);
        $tags        = explode(' ', $post->tags);
        $privacy     = 'public';
        $endPoint    = "https://www.googleapis.com/upload/youtube/v3/videos?uploadType=multipart&part=snippet,status";

        $metadata = [
            'snippet' => [
                'title'       => $title,
                'description' => $description,
                'tags'        => $tags,
            ],
            'status'  => [
                'privacyStatus'           => $privacy,
                'selfDeclaredMadeForKids' => $post->audience_type == status::FOR_KID ? true : false,
                'commentStatus'           => 'allowed',
            ],
        ];

        $videoData = file_get_contents($videoPath);
        $delimiter = uniqid();
        $body      = "--{$delimiter}\r\n";
        $body .= "Content-Type: application/json; charset=UTF-8\r\n\r\n";
        $body .= json_encode($metadata) . "\r\n";
        $body .= "--{$delimiter}\r\n";
        $body .= "Content-Type: video/mp4\r\n\r\n";
        $body .= $videoData . "\r\n";
        $body .= "--{$delimiter}--";

        $headers = [
            "Authorization: Bearer {$accessToken}",
            "Content-Type: multipart/related; boundary={$delimiter}",
            "Content-Length: " . strlen($body),
        ];
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $endPoint);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $body);

        $response = curl_exec($ch);

        if (curl_errno($ch)) {
            throw new \Exception("cURL Error: " . curl_error($ch));
        }

        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($httpCode !== 200) {
            throw new \Exception("YouTube API returned HTTP $httpCode: $response");
        }

        $result = json_decode($response, true);
        $postId = $result['id'];
        if (isset($postId) && $post->first_comment) {
            $this->postComment($socialAccount, $postId, $post->first_comment);
        }

        return $result['id'] ?? null;
    }

    public function postToLinkedin($post, $socialAccount) {
        $accessToken = $socialAccount->access_token;
        $authorUrn   = "urn:li:person:{$socialAccount->profile_id}";
        $content     = $post->content . ($post->tags ? "\n\n{$post->tags}" : '');
        $headers     = ['Content-Type: application/json', "Authorization: Bearer {$accessToken}"];

        $mediaAssets = [];
        $mediaType   = "NONE";

        foreach ($post->medias as $media) {
            try {
                if ($media->media_type === 'image') {
                    $imageUrl      = getImage(getFilePath('postMedia') . '/' . $media->media_path);
                    $mediaAssets[] = $this->uploadImageOnLinkedin($imageUrl, $authorUrn, $accessToken);
                    $mediaType     = "IMAGE";
                } else if ($media->media_type === 'video') {
                    $videoPath     = getImage(getFilePath('postMedia') . '/' . $media->media_path);
                    $mediaAssets[] = $this->uploadVideoOnLinkedin($videoPath, $authorUrn, $socialAccount);
                    $mediaType     = "VIDEO";
                }
            } catch (\Exception $e) {
                info("LinkedIn media upload error: " . $e->getMessage());
                continue;
            }
        }

        $endpoint = "https://api.linkedin.com/v2/ugcPosts";

        $mediaBlock = [];
        if ($mediaType === "IMAGE") {
            $mediaBlock = array_map(fn($asset) => [
                "status"      => "READY",
                "description" => ["text" => "Sample image"],
                "media"       => $asset,
                "title"       => ["text" => "Uploaded via API"],
            ], $mediaAssets);
        } else if ($mediaType === "VIDEO") {
            $mediaBlock = array_map(fn($asset) => [
                "status"      => "READY",
                "description" => ["text" => "Sample video"],
                "media"       => $asset,
                "title"       => ["text" => "Uploaded via API"],
            ], $mediaAssets);
        }

        $body = [
            "author"          => $authorUrn,
            "lifecycleState"  => "PUBLISHED",
            "specificContent" => [
                "com.linkedin.ugc.ShareContent" => [
                    "shareCommentary"    => [
                        "text" => $content,
                    ],
                    "shareMediaCategory" => $mediaType,
                    "media"              => $mediaBlock,
                ],
            ],
            "visibility"      => [
                "com.linkedin.ugc.MemberNetworkVisibility" => "PUBLIC",
            ],
        ];

        try {
            $response = CurlRequest::curlPostContent($endpoint, json_encode($body), $headers);
            $response = json_decode($response, true);

            if (isset($response['id']) && $post->first_comment) {
                $this->postComment($socialAccount, $response['id'], $post->first_comment);
            }
        } catch (\Exception $e) {
            info("LinkedIn post error: " . $e->getMessage());
            return false;
        }
    }

    public function uploadImageOnLinkedin($imageUrl, $authorUrn, $accessToken) {
        try {
            $endpoint = "https://api.linkedin.com/v2/assets?action=registerUpload";
            $headers  = ['Content-Type: application/json', "Authorization: Bearer {$accessToken}"];

            $body = [
                "registerUploadRequest" => [
                    "recipes"              => ["urn:li:digitalmediaRecipe:feedshare-image"],
                    "owner"                => $authorUrn,
                    "serviceRelationships" => [
                        [
                            "relationshipType" => "OWNER",
                            "identifier"       => "urn:li:userGeneratedContent",
                        ],
                    ],
                ],
            ];

            $response = CurlRequest::curlPostContent($endpoint, json_encode($body), $headers);
            $response = json_decode($response, true);

            $uploadUrl = $response['value']['uploadMechanism']['com.linkedin.digitalmedia.uploading.MediaUploadHttpRequest']['uploadUrl'];
            $assetUrn  = $response['value']['asset'];

            $imageData = file_get_contents($imageUrl);

            $uploadHeaders = [
                "Authorization: Bearer {$accessToken}",
                "Content-Type: image/jpeg",
            ];

            CurlRequest::curlPutBinary($uploadUrl, $imageData, $uploadHeaders);
            return $assetUrn;
        } catch (\Exception $e) {
            info("LinkedIn image upload error: " . $e->getMessage());
            return null;
        }
    }

    public function uploadVideoOnLinkedin($videoUrl, $authorUrn, $socialAccount) {
        $accessToken = $socialAccount->access_token;

        $registerUrl = "https://api.linkedin.com/v2/assets?action=registerUpload";
        $headers     = [
            "Authorization: Bearer {$accessToken}",
            "Content-Type: application/json",
        ];

        $registerBody = [
            "registerUploadRequest" => [
                "owner"                => $authorUrn,
                "recipes"              => ["urn:li:digitalmediaRecipe:feedshare-video"],
                "serviceRelationships" => [[
                    "relationshipType" => "OWNER",
                    "identifier"       => "urn:li:userGeneratedContent",
                ]],
            ],
        ];

        $registerResponse = CurlRequest::curlPostContent($registerUrl, json_encode($registerBody), $headers);
        $registerData     = json_decode($registerResponse, true);

        if (empty($registerData['value']['uploadMechanism'])) {
            throw new \Exception("Video registerUpload failed: " . $registerResponse);
        }

        $uploadUrl = $registerData['value']['uploadMechanism']['com.linkedin.digitalmedia.uploading.MediaUploadHttpRequest']['uploadUrl'];
        $asset     = $registerData['value']['asset'];

        $videoBinary = file_get_contents($videoUrl);
        if ($videoBinary === false) {
            throw new \Exception("Failed to load video from URL: {$videoUrl}");
        }

        $uploadHeaders = [
            "Authorization: Bearer {$accessToken}",
            "Content-Type: video/mp4",
            "Content-Length: " . strlen($videoBinary),
        ];

        CurlRequest::curlPutBinary($uploadUrl, $videoBinary, $uploadHeaders);
        return $asset;
    }

    public function postToTelegram($post, $socialAccount) {
        $accessToken = $socialAccount->access_token;
        $pageId      = $socialAccount->profile_id;
        $content     = $post->content . ($post->tags ? "\n\n{$post->tags}" : '');
        $medias      = [];

        foreach ($post->medias as $key => $media) {
            try {
                $mediaType = $media->media_type == "video" ? "video" : "photo";

                $imageUrl = getImage(getFilePath('postMedia') . '/' . $media->media_path);
                $videoUrl = getImage(getFilePath('postMedia') . '/' . $media->media_path);

                $item = [
                    'type'  => $mediaType,
                    'media' => $mediaType == "video" ? $videoUrl : $imageUrl,
                ];

                if ($key === 0) {
                    $item['caption'] = $content;
                }

                $medias[] = $item;
            } catch (\Exception $e) {
                info("Telegram media build error: " . $e->getMessage());
                continue;
            }
        }

        $endpoint = "https://api.telegram.org/bot{$accessToken}/sendMediaGroup";
        $body     = [
            'chat_id' => $pageId,
            'media'   => json_encode($medias),
        ];

        try {
            $response = CurlRequest::curlPostContent($endpoint, $body);
            $response = json_decode($response, true);

            return true;
        } catch (\Exception $e) {
            info("Facebook post error: " . $e->getMessage());
        }
    }

    public function postToTwitter($post, $socialAccount) {
        $accessToken = $socialAccount->access_token;
        $headers     = [
            'Authorization: Bearer ' . $accessToken,
            'Content-Type: application/json',
        ];
        $content  = $post->content . ($post->tags ? "\n\n{$post->tags}" : '');
        $mediaIds = [];

        foreach ($post->medias as $media) {
            try {
                if ($media->media_type === 'image') {
                    $mediaPath  = getImage(getFilePath('postMedia') . '/' . $media->media_path);
                    $mediaIds[] = (string) $this->uploadImageOnTwitter($mediaPath, $accessToken);
                } else if ($media->media_type === 'video') {
                    $videoPath = $media->media_path;
                    $postId    = $this->uploadVideoOnTwitter($videoPath, $accessToken, $content);
                    if ($postId) {
                        if ($post->first_comment != null) {
                            $this->postComment($socialAccount, $postId, $post->first_comment);
                        }
                    }
                    return true;
                }
            } catch (\Exception $e) {
                continue;
            }
        }

        $endPoint = "https://api.x.com/2/tweets";
        $body     = [
            'text'  => $content,
            'media' => [
                'media_ids' => array_map('strval', $mediaIds),
            ],
        ];

        try {
            $response = CurlRequest::curlPostContent($endPoint, json_encode($body), $headers);
            $response = json_decode($response, true);
            return $response['data']['id'] ?? '';
        } catch (\Exception $e) {
            info("Twitter post error: " . $e->getMessage());
        }
    }

    public function uploadImageOnTwitter($imageUrl, $accessToken) {
        $body = [
            'media'          => new \CURLFile($imageUrl),
            'media_category' => 'tweet_image',
        ];

        $headers = [
            "Authorization: Bearer {$accessToken}",
        ];
        $ch = curl_init("https://api.x.com/2/media/upload");
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $body);
        $response = curl_exec($ch);

        if (curl_errno($ch)) {
            throw new \Exception(curl_error($ch));
        }
        curl_close($ch);

        $response = json_decode($response, true);
        return $response['data']['id'];
    }

    public function uploadVideoOnTwitter($videoPath, $accessToken, $content) {
        try {
            $videoPath   = __DIR__ . '/../../../' . getFilePath('postMedia') . '/' . $videoPath;
            $totalBytes  = filesize($videoPath);
            $initialBody = [
                'media_type'     => 'video/mp4',
                'total_bytes'    => $totalBytes,
                'media_category' => 'tweet_video',
            ];
            $headers = [
                "Authorization: Bearer {$accessToken}",
                "Content-Type: application/json",
            ];
            $response = CurlRequest::curlPostContent("https://api.x.com/2/media/upload/initialize", json_encode($initialBody), $headers);
            $response = json_decode($response, true);

            $mediaId      = $response['data']['id'];
            $segmentIndex = 0;

            $chunkSize = 5 * 1024 * 1024;
            $handle    = fopen($videoPath, 'rb');

            while (!feof($handle)) {
                $chunk       = fread($handle, $chunkSize);
                $base64Chunk = base64_encode($chunk);
                $postFields  = [
                    'segment_index' => $segmentIndex,
                    'media'         => $base64Chunk,
                ];

                $appendHeaders = [
                    "Authorization: Bearer {$accessToken}",
                    "Content-Type: application/json",
                ];
                $result = CurlRequest::curlPostContent(
                    "https://api.x.com/2/media/upload/{$mediaId}/append",
                    json_encode($postFields),
                    $appendHeaders
                );

                $result = json_decode($result, true);

                if (isset($response['errors'])) {
                    throw new Exception('Append failed: ' . json_encode($response['errors']));
                }

                $segmentIndex++;
            }
            fclose($handle);

            $finalBody    = [];
            $finalHeaders = [
                "Authorization: Bearer {$accessToken}",
                "Content-Type: application/json",
            ];

            $finalResponse = CurlRequest::curlPostContent(
                "https://api.x.com/2/media/upload/{$mediaId}/finalize",
                json_encode($finalBody),
                $finalHeaders
            );

            $finalData = json_decode($finalResponse, true);

            $finalMediaId = $finalData['data']['id'];

            $processingState = $finalData['data']['processing_info']['state'] ?? 'pending';
            $checkAfterSecs  = $finalData['data']['processing_info']['check_after_secs'] ?? 1;

            $maxAttempts = 30;
            $attempts    = 0;

            while (in_array($processingState, ['pending', 'in_progress']) && $attempts < $maxAttempts) {
                sleep($checkAfterSecs);
                $statusHeaders = [
                    "Authorization: Bearer {$accessToken}",
                ];

                $statusResponse = CurlRequest::curlContent(
                    "https://api.x.com/2/media/upload?command=STATUS&media_id={$finalMediaId}",
                    $statusHeaders
                );

                $statusData = json_decode($statusResponse, true);

                $processingState = $statusData['data']['processing_info']['state'] ?? 'unknown';
                $checkAfterSecs  = $statusData['data']['processing_info']['check_after_secs'] ?? 1;

                $attempts++;

            }

            $endPoint = "https://api.x.com/2/tweets";
            $body     = [
                'text'  => $content,
                'media' => [
                    'media_ids' => [$mediaId],
                ],
            ];

            try {
                $response = CurlRequest::curlPostContent($endPoint, json_encode($body), $headers);
                $response = json_decode($response, true);
                return $response['data']['id'] ?? '';
            } catch (\Exception $e) {
                info("Twitter post error: " . $e->getMessage());
            }
        } catch (\Throwable $th) {
        }
    }

   public function postToTikTok($post, $socialAccount)
    {
        $accessToken = $socialAccount->access_token;
        $headers     = [
            'Authorization: Bearer ' . $accessToken,
            'Content-Type: application/json',
        ];
        
        $content    = $post->content . ($post->tags ? "\n\n{$post->tags}" : '');
        $mediaIds   = [];
        foreach ($post->medias as $media) {
            try {
                if ($media->media_type === 'image') {
                    
                    $imageFiles[] = getImage(getFilePath('postMedia') . '/' . $media->media_path);
                } else if ($media->media_type === 'video') {
                    $videoPath      = $media->media_path;
                    $postId         = $this->uploadVideoOnTiktok($videoPath, $accessToken, $content);
                    if ($postId) {
                        if ($post->first_comment != null) {
                            $this->postComment($socialAccount, $postId, $post->first_comment);
                        }
                    }
                    return true;
                }
            } catch (\Exception $e) {
                continue;
            }
        }
        $this->uploadImagesOnTiktok($imageFiles, $accessToken, $content);
    }

    public function uploadImagesOnTiktok($imageUrls, $accessToken, $content)
    {
        $url = 'https://open.tiktokapis.com/v2/post/publish/content/init/';
        $payload = [
            'post_info' => [
                'description'       => $content,
                'disable_comment'   => false,
                'privacy_level'     => 'PUBLIC_TO_EVERYONE',
                'auto_add_music'    => false
            ],
            'source_info' => [
                'source'            => 'PULL_FROM_URL',
                'photo_cover_index' => 1,
                'photo_images'      => $imageUrls
            ],
            'post_mode'         => 'DIRECT_POST',
            'media_type'        => 'PHOTO',
        ];

        $header = [
            "Authorization: Bearer {$accessToken}",
            "Content-Type: application/json"
        ];
    
        $response       = CurlRequest::curlPostContent($url,json_encode($payload),$header);
        $response       = json_decode($response, true);

        return $response;
    }
    
    public function uploadVideoOnTiktok(string $videoFile, string $accessToken, ?string $content = null): string
    {
        try {
            $videoUrl = getImage(getFilePath('postMedia') . '/' . $videoFile);
            if (!$videoUrl) {
                throw new \Exception('Invalid video URL');
            }
    
            $url = 'https://open.tiktokapis.com/v2/post/publish/video/init/';
            $headers = [
                "Authorization: Bearer {$accessToken}",
                "Content-Type: application/json",
            ];
    
            $payload = [
                'post_info' => [
                    'title'            => $content ?? 'My TikTok Video',
                    'privacy_level'    => 'PUBLIC_TO_EVERYONE',
                    'disable_duet'     => false,
                    'disable_comment'  => false,
                    'disable_stitch'   => false,
                ],
                'source_info' => [
                    'source'    => 'PULL_FROM_URL',
                    'video_url' => $videoUrl,
                ],
            ];
    
     
            $response = CurlRequest::curlPostContent(
                $url,
                json_encode($payload, JSON_UNESCAPED_SLASHES),
                $headers
            );
            
            if (!$response) {
                throw new \Exception('Empty response from TikTok');
            }
    
            $data = json_decode($response, true);

            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new \Exception('Invalid JSON response: ' . $response);
            }
    
            if (!empty($data['error']) && $data['error']['code'] !== 'ok') {
                throw new \Exception(
                    'TikTok API Error: ' . $data['error']['code'] . ' - ' . $data['error']['message']
                );
            }
    
            if (empty($data['data']['publish_id'])) {
                throw new \Exception('publish_id missing: ' . $response);
            }
            
            return $data['data']['publish_id'];
    
        } catch (\Throwable $e) {
            throw $e;
        }
    }

    private function postComment($socialAccount, $postId, $comment) {
        try {
            $headers = ['Content-Type: application/json'];
            if ($socialAccount->platform_id == Status::FACEBOOK) {
                $commentEndpoint = "https://graph.facebook.com/v23.0/{$postId}/comments";
                $commentBody     = [
                    'message'      => $comment,
                    'access_token' => $socialAccount->access_token,
                ];
                CurlRequest::curlPostContent($commentEndpoint, $commentBody, $headers);
            } else if ($socialAccount->platform_id == Status::INSTAGRAM) {
                $commentEndpoint = "https://graph.facebook.com/v23.0/{$postId}/comments";
                $commentBody     = [
                    'message'      => $comment,
                    'access_token' => $socialAccount->access_token,
                ];
                CurlRequest::curlPostContent($commentEndpoint, $commentBody, $headers);
            } else if ($socialAccount->platform_id == Status::LINKEDIN) {
                $commentEndpoint = "https://api.linkedin.com/v2/socialActions/{$postId}/comments";

                $commentBody = [
                    "actor" => "urn:li:person:{$socialAccount->profile_id}",
                    "message" => [
                        "text" => $comment,
                    ],
                ];

                $headers = [
                    "Authorization: Bearer {$socialAccount->access_token}",
                    "Content-Type: application/json",
                ];

                $response = CurlRequest::curlPostContent($commentEndpoint, json_encode($commentBody), $headers);
                $response = json_decode($response, true);
            } else if ($socialAccount->platform_id == Status::TWITTER) {
                $commentEndpoint = "https://api.twitter.com/2/tweets";

                $commentBody = [
                    'text'  => $comment,
                    'reply' => [
                        'in_reply_to_tweet_id' => (string) $postId,
                    ],
                ];

                $headers = [
                    "Authorization: Bearer {$socialAccount->access_token}",
                    "Content-Type: application/json",
                ];

                $response = CurlRequest::curlPostContent($commentEndpoint, json_encode($commentBody), $headers);
                $response = json_decode($response, true);
            } else {
                $commentEndpoint = "https://www.googleapis.com/youtube/v3/commentThreads?part=snippet";
                $commentBody     = [
                    'snippet' => [
                        'videoId'         => $postId,
                        'topLevelComment' => [
                            'snippet' => [
                                'textOriginal' => $comment,
                            ],
                        ],
                    ],
                ];

                $headers = [
                    'Authorization: Bearer ' . $socialAccount->access_token,
                    'Content-Type: application/json',
                ];
                $jsonBody = json_encode($commentBody);

                $response = CurlRequest::curlPostContent($commentEndpoint, $jsonBody, $headers);
                $response = json_decode($response);
            }

        } catch (\Throwable $th) {
            info($th->getMessage());
        }
    }

    public function refreshYoutubeAccessToken($socialAccount) {
        $clientId     = gs()->social_connect_credential->youtube->client_id;
        $clientSecret = gs()->social_connect_credential->youtube->client_secret;

        $refreshToken = $socialAccount->refresh_token;

        $response = CurlRequest::curlPostContent(
            'https://oauth2.googleapis.com/token',
            [
                'client_id'     => $clientId,
                'client_secret' => $clientSecret,
                'refresh_token' => $refreshToken,
                'grant_type'    => 'refresh_token',
            ]
        );

        $data = json_decode($response, true);

        $socialAccount               = SocialAccount::find($socialAccount->id);
        $socialAccount->access_token = $data['access_token'];
        $socialAccount->expires_at   = now()->addSeconds($data['expires_in']);
        $socialAccount->save();
        return $data['access_token'];
    }

    public function refreshLinkedinAccessToken($socialAccount) {
        $clientId     = gs()->social_connect_credential->linkedin->client_id;
        $clientSecret = gs()->social_connect_credential->linkedin->client_secret;

        $refreshToken = $socialAccount->refresh_token;

        $response = CurlRequest::curlPostContent(
            'https://www.linkedin.com/oauth/v2/accessToken',
            [
                'grant_type'    => "refresh_token",
                'refresh_token' => $refreshToken,
                'client_id'     => $clientId,
                'client_secret' => $clientSecret,
            ]
        );

        $data                        = json_decode($response, true);
        $socialAccount               = SocialAccount::find($socialAccount->id);
        $socialAccount->access_token = $data['access_token'];
        $socialAccount->expires_at   = now()->addSeconds($data['expires_in']);
        $socialAccount->save();
        return $data['access_token'];
    }

    public function refreshTwitterAccessToken($socialAccount) {
        $clientId     = gs()->social_connect_credential->twitter->client_id;
        $clientSecret = gs()->social_connect_credential->twitter->client_secret;
        $refreshToken = $socialAccount->refresh_token;

        $postFields = [
            'grant_type'    => 'refresh_token',
            'refresh_token' => $refreshToken,
            'client_id'     => $clientId,
        ];

        $headers = [
            'Authorization: Basic ' . base64_encode($clientId . ':' . $clientSecret),
            'Content-Type: application/x-www-form-urlencoded;charset=UTF-8',
        ];

        $response = CurlRequest::curlPostContent('https://api.twitter.com/2/oauth2/token', $postFields, $headers);

        $data = json_decode($response, true);

        $socialAccount                = SocialAccount::find($socialAccount->id);
        $socialAccount->access_token  = $data['access_token'];
        $socialAccount->expires_at    = now()->addSeconds($data['expires_in']);
        $socialAccount->refresh_token = $data['refresh_token'] ?? $refreshToken;
        $socialAccount->save();

        return $data['access_token'];
    }
    
    public function refreshTiktokAccessToken($socialAccount)
    {
        $clientId     = gs()->social_connect_credential->tiktok->client_id;
        $clientSecret = gs()->social_connect_credential->tiktok->client_secret;
        $refreshToken = $socialAccount->refresh_token;
    
        $body = http_build_query([
            'client_key'    => $clientId,
            'client_secret' => $clientSecret,
            'grant_type'    => 'refresh_token',
            'refresh_token' => $refreshToken,
        ]);
    
        $response = CurlRequest::curlPostContent(
            'https://open.tiktokapis.com/v2/oauth/token/',
            $body,
            [
                'Content-Type: application/x-www-form-urlencoded',
                'Cache-Control: no-cache',
            ]
        );
    
        $data = json_decode($response, true);
        
        $socialAccount                = SocialAccount::find($socialAccount->id);
        $socialAccount->access_token  = $data['access_token'];
        $socialAccount->expires_at    = now()->addSeconds($data['expires_in']);
        $socialAccount->refresh_token = $data['refresh_token'] ?? $refreshToken;
        $socialAccount->save();
    }
}
