<?php

namespace App\Http\Controllers\User;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Models\DeviceToken;
use App\Models\Plan;
use App\Models\Platform;
use App\Models\Post;
use App\Models\ReferralSetting;
use App\Models\SocialAccount;
use App\Models\Subscription;
use App\Models\Transaction;
use App\Models\User;
use App\Models\UserNotification;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rule;

class UserController extends Controller {
    public function home() {
        $user                         = auth()->user();
        $pageTitle                    = 'Dashboard';
        $posts                        = Post::where('user_id', $user->id);
        $widget['total_post']         = (clone $posts)->where('user_id', $user->id)->count();
        $widget['connected_account']  = SocialAccount::active()->where('user_id', $user->id)->count();
        $widget['scheduled_post']     = (clone $posts)->where('status', Status::SCHEDULE)->count();
        $widget['available_schedule'] = $user->total_schedule - $user->used_schedule;
        $draftPosts                   = (clone $posts)->where('status', Status::DRAFT)->latest()->take(4)->get();

        $platformData = [];
        $platforms    = Platform::active()
            ->with(['accounts' => function ($q) use ($user) {
                $q->where('user_id', $user->id);
            }])->get();

        foreach ($platforms as $platform) {
            $hasAccounts = $platform->accounts->count() > 0;

            if ($platform->id == Status::FACEBOOK) {
                $platformData['facebook'] = [
                    'hasAccounts' => $hasAccounts,
                ];
            } else if ($platform->id == Status::INSTAGRAM) {
                $platformData['instagram'] = [
                    'hasAccounts' => $hasAccounts,
                ];
            } else if ($platform->id == Status::YOUTUBE) {
                $platformData['youtube'] = [
                    'hasAccounts' => $hasAccounts,
                ];
            } else if ($platform->id == Status::TWITTER) {
                $platformData['twitter'] = [
                    'hasAccounts' => $hasAccounts,
                ];
            } else if ($platform->id == Status::LINKEDIN) {
                $platformData['linkedIn'] = [
                    'hasAccounts' => $hasAccounts,
                ];
            } else if ($platform->id == Status::TELEGRAM) {
                $platformData['telegram'] = [
                    'hasAccounts' => $hasAccounts,
                ];
            } else if ($platform->id == Status::TIKTOK) {
                $platformData['tiktok'] = [
                    'hasAccounts' => $hasAccounts,
                ];
            }
        }

        $monthlyPosts = (clone $posts)->where(function ($query) {
            $query->whereNotNull('schedule_start_date')
                ->orWhereNotNull('schedule_end_date');
        })
            ->whereYear('created_at', Carbon::now()->year)
            ->selectRaw('MONTH(created_at) as month, COUNT(*) as total')
            ->groupBy('month')
            ->pluck('total', 'month');

        $postAnalytics = [];
        for ($i = 1; $i <= 12; $i++) {
            $postAnalytics[] = $monthlyPosts[$i] ?? 0;
        }

        return view('Template::user.dashboard', compact('pageTitle', 'widget', 'postAnalytics', 'draftPosts', 'platformData'));
    }

    public function depositHistory(Request $request) {
        $pageTitle = 'Payment History';
        $deposits  = auth()->user()->deposits()->searchable(['trx'])->with(['gateway'])->orderBy('id', 'desc')->paginate(getPaginate());
        return view('Template::user.deposit_history', compact('pageTitle', 'deposits'));
    }

    public function userData() {
        $user = auth()->user();

        if ($user->profile_complete == Status::YES) {
            return to_route('user.home');
        }

        $pageTitle  = 'User Data';
        $info       = json_decode(json_encode(getIpInfo()), true);
        $mobileCode = isset($info['code']) ? implode(',', $info['code']) : '';
        $countries  = json_decode(file_get_contents(resource_path('views/partials/country.json')));

        return view('Template::user.user_data', compact('pageTitle', 'user', 'countries', 'mobileCode'));
    }

    public function userDataSubmit(Request $request) {

        $user = auth()->user();

        if ($user->profile_complete == Status::YES) {
            return to_route('user.home');
        }

        $countryData  = (array) json_decode(file_get_contents(resource_path('views/partials/country.json')));
        $countryCodes = implode(',', array_keys($countryData));
        $mobileCodes  = implode(',', array_column($countryData, 'dial_code'));
        $countries    = implode(',', array_column($countryData, 'country'));

        $request->validate([
            'country_code' => 'required|in:' . $countryCodes,
            'country'      => 'required|in:' . $countries,
            'mobile_code'  => 'required|in:' . $mobileCodes,
            'username'     => 'required|unique:users|min:6',
            'mobile'       => ['required', 'regex:/^([0-9]*)$/', Rule::unique('users')->where('dial_code', $request->mobile_code)],
        ]);

        if (preg_match("/[^a-z0-9_]/", trim($request->username))) {
            $notify[] = ['info', 'Username can contain only small letters, numbers and underscore.'];
            $notify[] = ['error', 'No special character, space or capital letters in username.'];
            return back()->withNotify($notify)->withInput($request->all());
        }

        $user->country_code = $request->country_code;
        $user->mobile       = $request->mobile;
        $user->username     = $request->username;

        $user->address      = $request->address;
        $user->city         = $request->city;
        $user->state        = $request->state;
        $user->zip          = $request->zip;
        $user->country_name = isset($request->country) ? $request->country : '';
        $user->dial_code    = $request->mobile_code;

        $user->profile_complete = Status::YES;
        $user->save();

        return to_route('user.home');
    }

    public function addDeviceToken(Request $request) {

        $validator = Validator::make($request->all(), [
            'token' => 'required',
        ]);

        if ($validator->fails()) {
            return ['success' => false, 'errors' => $validator->errors()->all()];
        }

        $deviceToken = DeviceToken::where('token', $request->token)->first();

        if ($deviceToken) {
            return ['success' => true, 'message' => 'Already exists'];
        }

        $deviceToken          = new DeviceToken();
        $deviceToken->user_id = auth()->user()->id;
        $deviceToken->token   = $request->token;
        $deviceToken->is_app  = Status::NO;
        $deviceToken->save();

        return ['success' => true, 'message' => 'Token saved successfully'];
    }

    public function downloadAttachment($fileHash) {
        $filePath  = decrypt($fileHash);
        $extension = pathinfo($filePath, PATHINFO_EXTENSION);
        $title     = slug(gs('site_name')) . '- attachments.' . $extension;
        try {
            $mimetype = mime_content_type($filePath);
        } catch (\Exception $e) {
            $notify[] = ['error', 'File does not exists'];
            return back()->withNotify($notify);
        }
        header('Content-Disposition: attachment; filename="' . $title);
        header("Content-Type: " . $mimetype);
        return readfile($filePath);
    }

    public function notification() {
        $pageTitle     = 'Notifications';
        $notifications = UserNotification::where('user_id', auth()->id())->orderBy('id', 'desc')->paginate(getPaginate());
        return view('Template::user.notification', compact('pageTitle', 'notifications'));
    }
    public function notificationRead($id) {
        $notification          = UserNotification::findOrFail($id);
        $notification->is_read = Status::YES;
        $notification->save();
        $url = $notification->click_url;
        if ($url == '#') {
            $url = url()->previous();
        }
        return redirect($url);
    }

    public function subscribePlan($id, $interval = 0) {
        $plan = Plan::active()->findOrFail($id);

        $user           = auth()->user();
        $pendingPayment = $user->deposits()->where('status', Status::PAYMENT_PENDING)->count();
        if ($pendingPayment > 0) {
            $notify[] = ['error', 'Already one payment in pending. Please Wait'];
            return back()->withNotify($notify);
        }

        $days     = $interval ? 365 : 30;
        $price    = $interval ? $plan->yearly_price : $plan->monthly_price;
        $credit   = $interval ? $plan->yearly_credit : $plan->monthly_credit;
        $type     = $interval ? Status::YEARLY_PLAN : Status::MONTHLY_PLAN;
        $channel  = $interval ? $plan->monthly_total_channel : $plan->yearly_total_channel;
        $schedule = $interval ? $plan->monthly_total_post_schedule : $plan->yearly_total_post_schedule;

        $subscription                 = new Subscription();
        $subscription->user_id        = auth()->id();
        $subscription->plan_id        = $plan->id;
        $subscription->price          = $price;
        $subscription->expired_date   = now()->addDays($days);
        $subscription->total_channel  = $channel;
        $subscription->total_schedule = $schedule;
        $subscription->plan_type      = $type;
        $subscription->status         = Status::PAYMENT_INITIATE;
        $subscription->save();

        session(['subscription_id' => $subscription->id]);
        return redirect()->route('user.deposit.index');
    }

    public function referrals() {
        $pageTitle = 'Referrals';
        $user      = User::where('id', auth()->id())->with('allReferrals')->firstOrFail();
        $maxLevel  = ReferralSetting::max('level');
        return view('Template::user.referrals', compact('pageTitle', 'user', 'maxLevel'));
    }

    public function transactions() {
        $pageTitle    = 'Transactions';
        $transactions = Transaction::searchable(['trx'])->where('user_id', auth()->id())->latest()->paginate(getPaginate());
        return view('Template::user.transactions', compact('pageTitle', 'transactions'));
    }
}
