<?php

namespace App\Http\Controllers\User;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Lib\MetaPublisher;
use App\Models\Post;
use App\Models\PostMedia;
use App\Models\SocialAccount;
use App\Rules\FileTypeValidate;
use Illuminate\Http\Request;
use OpenAI;

class PostController extends Controller {
    public function index() {
        $pageTitle = 'Manage Posts';
        $posts     = $this->postData();
        return view("Template::user.post.index", compact('pageTitle', 'posts'));
    }
    public function scheduled() {
        $pageTitle = 'Scheduled Posts';
        $posts     = $this->postData($scope = 'schedule');

        return view("Template::user.post.index", compact('pageTitle', 'posts'));
    }
    public function drafted() {
        $pageTitle = 'Drafted Posts';
        $posts     = $this->postData($scope = 'draft');

        return view("Template::user.post.index", compact('pageTitle', 'posts'));
    }

    public function published() {
        $pageTitle = 'Published Posts';
        $posts     = $this->postData($scope = 'published');

        return view("Template::user.post.index", compact('pageTitle', 'posts'));
    }

    public function postData($scope = null) {
        if ($scope) {
            $posts = Post::$scope();
        } else {
            $posts = Post::query();
        }
        $posts = $posts->with('medias')->where('user_id', auth()->id())->latest()->paginate(getPaginate());
        return $posts;
    }

    public function create($id = 0) {
        $pageTitle = 'Create Posts';
        $accounts  = SocialAccount::active()->with('platform')->where('user_id', auth()->id())->get();
        $post      = null;
        if ($id != 0) {
            $post = Post::with('medias')->where('user_id', auth()->id())->findOrFail($id);
        }
        $user = auth()->user();
        return view("Template::user.post.create", compact('pageTitle', 'accounts', 'post', 'user'));
    }

    public function store(Request $request, $id = 0) {
        $validation = validator($request->all(), [
            'platform_type'    => 'required',
            'title'            => 'required_if:platform_type,video|max:255',
            'content'          => 'required|string',
            'post_type'        => 'required_if:submit_type,scheduled',
            'submit_type'      => 'required',
            'schedule_date'    => 'required_if:submit_type,scheduled',
            'schedule_time'    => 'required_if:submit_type,scheduled',
            'selected_account' => 'required',
            'media.*'          => ['required', new FileTypeValidate(['jpeg', 'jpg', 'png','webp', 'mp4', 'mkv'])],
        ]);

        if ($validation->fails()) {
            $notify[] = ['error', $validation->errors()->first()];
            return responseError('validation_error', $notify);
        }

        if (!activePlan()) {
            $notify[] = 'You need to subscribe to a plan to create posts';
            return responseError('validation_error', $notify);
        }

        if ($request->submit_type == Status::SCHEDULE) {
            $user = activePlan();
            if (!$user || $user->total_schedule <= $user->used_schedule) {
                $notify[] = 'You have reached the limit of scheduled posts';
                return responseError('validation_error', $notify);
            }
        }

        try {
            $user = auth()->user();
            if ($id != 0) {
                $post = Post::findOrFail($id);
                if (!empty($request->removed_medias)) {
                    $oldFileIds = explode(',', $request->removed_medias);
                    foreach ($oldFileIds as $mediaId) {
                        $media = PostMedia::find($mediaId);
                        if ($media) {
                            fileManager()->removeFile(getFilePath('postMedia') . '/' . $media->media_path);
                            $media->delete();
                        }
                    }
                }

                if ($request->submit_type == 'schedule') {
                    if ($post->status != Status::SCHEDULE) {
                        $user->used_schedule = $user->used_schedule + 1;
                        $user->save();
                    }
                }

                $message = "Post updated successfully";
            } else {
                $post = new Post();
                if ($request->submit_type == 'schedule') {
                    $user->used_schedule = $user->used_schedule + 1;
                    $user->save();
                }
                $message = "Post created successfully";
            }

            $post->user_id       = $user->id;
            $post->account_ids   = $request->selected_account;
            $post->title         = $request->title;
            $post->content       = $request->content;
            $post->tags          = $request->tags;
            $post->media_type    = $request->media_type;
            $post->post_type     = $request->post_type ?? 0;
            $post->who_can_see   = $request->who_can_see;
            $post->first_comment = $request->first_comment;
            $post->audience_type = $request->audience_type == Status::FOR_KID ? Status::FOR_KID : Status::NOT_FOR_KID;

            if ($request->submit_type != 'publish') {
                if ($request->schedule_date && $request->schedule_time) {
                    $scheduleDates             = explode(' - ', $request->schedule_date);
                    $post->schedule_start_date = $scheduleDates[0];
                    $post->schedule_end_date   = $scheduleDates[1];

                    $post->schedule_time = $request->schedule_time;
                }
            }

            if ($request->submit_type == 'schedule') {
                $post->status = Status::SCHEDULE;
            } else if ($request->submit_type == 'publish') {
                $post->status = Status::PUBLISH;
            } else {
                $post->status = Status::DRAFT;
            }

            $post->save();

            if ($request->hasFile('media')) {
                foreach ($request->file('media') as $file) {
                    $media             = new PostMedia();
                    $media->post_id    = $post->id;
                    $media->media_type = $request->media_type == status::IMAGE ? status::IMAGE : status::VIDEO;
                    $media->media_path = fileUploader($file, getFilePath('postMedia'));
                    $media->save();
                }
            }

            if ($post->status == Status::PUBLISH) {
                $publisher = new MetaPublisher($post);
                $response  = $publisher->publish();
                if (isset($response['status']) && $response['status'] == 'error') {
                    $notify[] = ['error', $response['message']];
                    return responseError('error', $notify);
                }
            }

            $notify[] = ['success', $message];
            return responseSuccess('success', $notify);
        } catch (\Throwable $th) {
            $notify[] = ['error', $th->getMessage()];
            return responseError('error', $notify);
        }
    }

    public function hashtagGenerate(Request $request) {
        $prompt = <<<PROMPT
        Generate exactly 5 relevant, popular hashtags for this topic: "$request->prompt".
        Follow these rules strictly:
        1. Only output the 5 hashtags, nothing else
        2. Format as: #OneWordHashtag1 #SecondHashtag2 #Third3 #Fourth4 #Fifth5
        3. No explanations, titles, or additional text
        4. Use current trending terms when applicable
        5. Ensure all hashtags are single words or combined words (no spaces)
        6. Make them highly relevant to the input topic
        7. Include 1-2 broad interest hashtags and 3-4 niche-specific ones
        8. Give the tags in PascalCase format
        PROMPT;

        try {
            $client   = OpenAI::factory()->withApiKey(gs()->openai_key)->make();
            $response = $client->chat()->create([
                'model'    => gs()->model->name,
                'messages' => [
                    [
                        'role'    => 'system',
                        'content' => "Generate 5 highly relevant hashtags for: {$request->prompt}\n\n" .
                        "Format requirements:\n" .
                        "- Only output #hashtag1 #hashtag2 #hashtag3 #hashtag4 #hashtag5\n" .
                        "- All hashtags must be single compound words (no spaces)\n" .
                        "- Mix 2 popular hashtags with 3 niche-specific ones\n" .
                        "- Use current trending terms when applicable\n" .
                        "- Never include any other text or explanations",
                    ],
                    [
                        'role'    => 'user',
                        'content' => $prompt,
                    ],
                ],
                'temperature' => 0.5,
                'max_tokens'  => 60,
            ]);

            $hashTags = explode(' ', trim($response->choices[0]->message->content));

            return responseSuccess('hashTags', [], ['hashTags' => $hashTags]);
        } catch (\Exception $e) {
            $message = $e->getMessage();
            return responseError($message);
        }
    }

    public function contentGenerate(Request $request) {
        $prompt = <<<PROMPT
        Generate only post content for this topic: "$request->prompt".
        Follow these rules strictly:
        1. Only output the content, nothing else
        2. No explanations, titles, or additional text
        3. Use current trending terms when applicable
        4. Ensure the content is relevant to the input topic
        5. Make the content realstic and engaging
        6. Keep the content under 60 words
        7. Do not add any hashtags
        PROMPT;

        try {
            $client   = OpenAI::factory()->withApiKey(gs()->openai_key)->make();
            $response = $client->chat()->create([
                'model'       => gs()->model->name,
                'messages'    => [
                    [
                        'role'    => 'system',
                        'content' => "You are a social media caption writer.
                            Your job is to create short, catchy, and trending captions for posts.
                            Do NOT write paragraphs or long explanations.
                            The output must look like a real social media caption:
                            - One to three short lines max
                            - Can include emojis, hashtags, or trending slang
                            - Keep it under 100 words
                            - Must be engaging and human-like
                            - Never include any hashtags",

                    ],
                    [
                        'role'    => 'user',
                        'content' => $prompt,
                    ],
                ],
                'temperature' => 0.5,
                'max_tokens'  => 60,
            ]);

            $content = trim($response->choices[0]->message->content);
            return responseSuccess('content', [], ['content' => $content]);
        } catch (\Exception $e) {
            $message = $e->getMessage();
            return responseError($message);
        }
    }

    public function delete($id) {
        $post = Post::with('medias')->findOrFail($id);
        foreach ($post->medias as $media) {
            fileManager()->removeFile(getFilePath('postMedia') . '/' . $media->media_path);
            $media->delete();
        }

        $post->delete();
        $notify[] = ['success', 'Post deleted successfully'];
        return back()->withNotify($notify);
    }
}
